package com.vmware.samples.chassisui {

import flash.events.Event;

/**
 * Generic event for returning wizard data once the wizard is complete.
 *
 * <p/>This can be re-used in any plugin code using wizards, or you could customize it
 * for your application.
 *
 * <p/>Example using WizardCompletionEvent:
 * <listing version="3.0">
 *    // In the wizard class
 *    ...
 *    // Event listener for the click in the Finish button.
 *    private function onWizardCompleted(event:DialogEvent):void {
 *       dispatchEvent(WizardCompletionEvent.newInstance(data, "ChassisWizard"));
 *    }
 *    ...
 *
 *    // In some global event manager class (not the action command code)
 *
 *   [EventHandler(name="{com.vmware.samples.chassisui.WizardCompletionEvent.EVENT_ID}")]
 *   public function onWizardCompleted(event:WizardCompletionEvent):void {
 *      if ((event.wizardName == "ChassisWizard") && (event.wizardData is ChassisInfo)) {
 *         _proxy.createChassis(event.wizardData as ChassisInfo, onCreateChassisComplete);
 *      }
 *   }
 * </listing>
 */
public class WizardCompletionEvent extends Event {
   public static const EVENT_ID:String = "wizardCompletionEventId";

   /**
    * The data to return.
    */
   public var wizardData:Object;

   /**
    * Some wizard identification, to be sure to process the right event.
    */
   public var wizardName:String;

   /**
    * Constructor
    */
   public function WizardCompletionEvent() {
      super(EVENT_ID);
   }

   /**
    * Returns a new instance.
    *
    * @param wizardData The data to return
    * @param wizardName Some wizard identification
    * @return a WizardCompletionEvent
    */
   public static function newInstance(wizardData:Object, wizardName:String)
         :WizardCompletionEvent {
      var instance:WizardCompletionEvent = new WizardCompletionEvent();
      instance.wizardData = wizardData;
      instance.wizardName = wizardName;
      return instance;
   }

   /**
    * Duplicates the current instance.
    */
   public override function clone():Event {
      return WizardCompletionEvent.newInstance(this.wizardData, this.wizardName);
   }
}
}